<?php

namespace Drupal\test_custom_entity\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\test_custom_entity\Entity\Bundle\TestCustomEntityBundle;
use Drupal\test_custom_entity\TestCustomEntityInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the test custom entity entity class.
 *
 * @ContentEntityType(
 *   id = "test_custom_entity",
 *   label = @Translation("Test custom entity"),
 *   label_collection = @Translation("Test custom entities"),
 *   label_singular = @Translation("test custom entity"),
 *   label_plural = @Translation("test custom entities"),
 *   label_count = @PluralTranslation(
 *     singular = "@count test custom entities",
 *     plural = "@count test custom entities",
 *   ),
 *   bundle_label = @Translation("Test custom entity type"),
 *   handlers = {
 *     "list_builder" = "Drupal\test_custom_entity\TestCustomEntityListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\test_custom_entity\Form\TestCustomEntityForm",
 *       "edit" = "Drupal\test_custom_entity\Form\TestCustomEntityForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     }
 *   },
 *   base_table = "test_custom_entity",
 *   revision_table = "test_custom_entity_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer test custom entity types",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "bundle" = "bundle",
 *     "label" = "id",
 *     "owner" = "uid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "collection" = "/admin/content/test-custom-entity",
 *     "add-form" = "/test-custom-entity/add/{bundle}",
 *     "add-page" = "/test-custom-entity/add",
 *     "canonical" = "/test-custom-entity/{test_custom_entity}",
 *     "edit-form" = "/test-custom-entity/{test_custom_entity}/edit",
 *     "delete-form" = "/test-custom-entity/{test_custom_entity}/delete",
 *   },
 *   field_ui_base_route = "entity.test_custom_entity_type.edit_form",
 * )
 */
class TestCustomEntity extends RevisionableContentEntityBase implements TestCustomEntityInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getBundle() {
    return $this->get('bundle')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setBundle($bundle) {
    $this->set('bundle', $bundle);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(static::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the test custom entity was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the test custom entity was last edited.'));

    return $fields;
  }

  public static function bundleFieldDefinitions(EntityTypeInterface $entity_type, $bundle, array $base_field_definitions) {
    // Fields to be shared by all bundles go here.
    $definitions = [];

    // Then add fields from the bundle in the current instance.
    $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo('test_custom_entity');
    foreach ($bundles as $key => $values) {
      if ($bundle == $key) {
        /** @var TestCustomEntityBundle $qualified_class */
        $qualified_class = '\\' . $values['class'];
        $definitions = $qualified_class::bundleFieldDefinitions($entity_type, $bundle, []);
      }
    }
    return $definitions;
  }

}
